<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Http\Requests\ExpenseStoreRequest;
use App\Http\Requests\ExpenseUpdateRequest;
use App\Models\Expense;
use Illuminate\Http\Request;

class ExpenseController extends Controller
{
    public function index(Request $request)
    {
        $provider = auth()->user()->provider;
        
        $query = $provider->expenses()
            ->with(['expenseType', 'supplier'])
            ->when($request->search, function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('reference', 'like', "%{$search}%")
                        ->orWhere('notes', 'like', "%{$search}%")
                        ->orWhereHas('expenseType', function ($typeQuery) use ($search) {
                            $typeQuery->where('name', 'like', "%{$search}%");
                        });
                });
            })
            ->when($request->expense_type_id, function ($q, $typeId) {
                $q->where('expense_type_id', $typeId);
            })
            ->when($request->payment_method, function ($q, $method) {
                $q->where('payment_method', $method);
            })
            ->when($request->supplier_id, function ($q, $supplierId) {
                $q->where('supplier_id', $supplierId);
            })
            ->when($request->date_from, function ($q, $dateFrom) {
                $q->where('expense_date', '>=', $dateFrom);
            })
            ->when($request->date_to, function ($q, $dateTo) {
                $q->where('expense_date', '<=', $dateTo);
            })
            ->orderBy('expense_date', 'desc')
            ->orderBy('id', 'desc');

        $expenses = $query->paginate(15);
        
        $expenseTypes = $provider->expenseTypes()->where('is_active', true)->orderBy('name')->get();
        $suppliers = $provider->suppliers()->orderBy('name')->get();

        return view('provider.expenses.index', compact('expenses', 'expenseTypes', 'suppliers'));
    }

    public function create()
    {
        $provider = auth()->user()->provider;
        
        $expenseTypes = $provider->expenseTypes()->where('is_active', true)->orderBy('name')->get();
        $suppliers = $provider->suppliers()->orderBy('name')->get();
        
        return view('provider.expenses.create', compact('expenseTypes', 'suppliers'));
    }

    public function store(ExpenseStoreRequest $request)
    {
        $provider = auth()->user()->provider;
        
        $data = $request->validated();
        $data['provider_id'] = $provider->id;
        
        $provider->expenses()->create($data);

        return redirect(route('provider.finances.index') . '?tab=depenses')
            ->with('success', 'Dépense créée avec succès.');
    }

    public function edit($expense_uuid)
    {
        $expense = $this->findExpenseByUuid($expense_uuid);
        $provider = auth()->user()->provider;
        
        $expenseTypes = $provider->expenseTypes()->where('is_active', true)->orderBy('name')->get();
        $suppliers = $provider->suppliers()->orderBy('name')->get();
        
        return view('provider.expenses.edit', compact('expense', 'expenseTypes', 'suppliers'));
    }

    public function update(ExpenseUpdateRequest $request, $expense_uuid)
    {
        $expense = $this->findExpenseByUuid($expense_uuid);
        
        $expense->update($request->validated());

        return redirect(route('provider.finances.index') . '?tab=depenses')
            ->with('success', 'Dépense modifiée avec succès.');
    }

    public function destroy($expense_uuid)
    {
        $expense = $this->findExpenseByUuid($expense_uuid);
        
        $expense->delete();

        return redirect(route('provider.finances.index') . '?tab=depenses')
            ->with('success', 'Dépense supprimée avec succès.');
    }

    private function findExpenseByUuid($expense_uuid)
    {
        $provider = auth()->user()->provider;
        
        return $provider->expenses()
            ->where('uuid', $expense_uuid)
            ->firstOrFail();
    }
}