<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Http\Requests\ExpenseTypeStoreRequest;
use App\Http\Requests\ExpenseTypeUpdateRequest;
use App\Models\ExpenseType;
use Illuminate\Http\Request;

class ExpenseTypeController extends Controller
{
    public function index(Request $request)
    {
        $provider = auth()->user()->provider;
        
        $query = $provider->expenseTypes()
            ->when($request->search, function ($q, $search) {
                $q->where('name', 'like', "%{$search}%");
            })
            ->when($request->has('status'), function ($q) use ($request) {
                if ($request->status === 'active') {
                    $q->where('is_active', true);
                } elseif ($request->status === 'inactive') {
                    $q->where('is_active', false);
                }
            })
            ->orderBy('is_active', 'desc')
            ->orderBy('name', 'asc');

        $expenseTypes = $query->paginate(15);

        return view('provider.expense-types.index', compact('expenseTypes'));
    }

    public function create()
    {
        return view('provider.expense-types.create');
    }

    public function store(ExpenseTypeStoreRequest $request)
    {
        $provider = auth()->user()->provider;
        
        $data = $request->validated();
        $data['is_active'] = $request->has('is_active');
        
        $expenseType = $provider->expenseTypes()->create($data);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'expenseType' => $expenseType
            ]);
        }

        return redirect(route('provider.finances.index') . '#types-depense')
            ->with('success', 'Type de dépense créé avec succès.');
    }

    public function edit($expense_type_uuid)
    {
        $expenseType = $this->findExpenseTypeByUuid($expense_type_uuid);
        
        return view('provider.expense-types.edit', compact('expenseType'));
    }

    public function update(ExpenseTypeUpdateRequest $request, $expense_type_uuid)
    {
        $expenseType = $this->findExpenseTypeByUuid($expense_type_uuid);
        
        $data = $request->validated();
        $data['is_active'] = $request->has('is_active');
        
        $expenseType->update($data);

        return redirect()->route('provider.finances.index', ['tab' => 'types-depense'])
            ->with('success', 'Type de dépense modifié avec succès.');
    }

    public function destroy($expense_type_uuid)
    {
        $expenseType = $this->findExpenseTypeByUuid($expense_type_uuid);
        
        if ($expenseType->expenses()->exists()) {
            return redirect()->route('provider.finances.index', ['tab' => 'types-depense'])
                ->with('error', 'Impossible de supprimer ce type de dépense car il est utilisé par des dépenses existantes.');
        }
        
        $expenseType->delete();

        return redirect()->route('provider.finances.index', ['tab' => 'types-depense'])
            ->with('success', 'Type de dépense supprimé avec succès.');
    }

    private function findExpenseTypeByUuid($expense_type_uuid)
    {
        $provider = auth()->user()->provider;
        
        return $provider->expenseTypes()
            ->where('uuid', $expense_type_uuid)
            ->firstOrFail();
    }
}