<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Expense;
use App\Models\ExpenseType;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class FinancesController extends Controller
{
    public function index()
    {
        $provider = Auth::user()->provider;
        
        // Default to current month
        $startDate = request('start_date') 
            ? Carbon::createFromFormat('Y-m-d', request('start_date')) 
            : Carbon::now()->startOfMonth();
        $endDate = request('end_date') 
            ? Carbon::createFromFormat('Y-m-d', request('end_date')) 
            : Carbon::now()->endOfMonth();

        // Get financials data with filters
        $paymentsQuery = Payment::where('provider_id', $provider->id)
            ->whereBetween('payment_date', [$startDate, $endDate])
            ->with('client');
        
        if (request('client_id')) {
            $paymentsQuery->where('client_id', request('client_id'));
        }
        
        if (request('method')) {
            $paymentsQuery->where('method', request('method'));
        }
        
        if (request('date_from')) {
            $paymentsQuery->where('payment_date', '>=', request('date_from'));
        }
        
        if (request('date_to')) {
            $paymentsQuery->where('payment_date', '<=', request('date_to'));
        }
        
        $payments = $paymentsQuery->orderBy('payment_date', 'desc')->orderBy('id', 'desc')->paginate(15);

        $expensesQuery = Expense::where('provider_id', $provider->id)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->with(['expenseType', 'supplier']);
        
        if (request('expense_type_id')) {
            $expensesQuery->where('expense_type_id', request('expense_type_id'));
        }
        
        if (request('supplier_id')) {
            $expensesQuery->where('supplier_id', request('supplier_id'));
        }
        
        if (request('payment_method')) {
            $expensesQuery->where('payment_method', request('payment_method'));
        }
        
        if (request('expense_reference')) {
            $expensesQuery->where('reference', 'like', '%' . request('expense_reference') . '%');
        }
        
        if (request('expense_notes')) {
            $expensesQuery->where('notes', 'like', '%' . request('expense_notes') . '%');
        }
        
        $expenses = $expensesQuery->latest('expense_date')->latest('id')->paginate(15);

        $totalPayments = Payment::where('provider_id', $provider->id)
            ->whereBetween('payment_date', [$startDate, $endDate])
            ->sum('amount');

        $totalExpenses = Expense::where('provider_id', $provider->id)
            ->whereBetween('expense_date', [$startDate, $endDate])
            ->sum('amount');

        $balanceNet = $totalPayments - $totalExpenses;
        
        $expenseTypesQuery = ExpenseType::where('provider_id', $provider->id);
        
        if (request('type_search')) {
            $expenseTypesQuery->where('name', 'like', '%' . request('type_search') . '%');
        }
        
        $expenseTypes = $expenseTypesQuery->withCount('expenses')->orderBy('created_at', 'desc')->get();
        
        $allExpenseTypes = ExpenseType::where('provider_id', $provider->id)->withCount('expenses')->orderBy('created_at', 'desc')->get();
        
        $clients = $provider->clients()->orderBy('first_name')->get();
        $suppliers = $provider->suppliers()->orderBy('name')->get();
        
        $currency = $provider->generalSettings['currency'] ?? 'TND';

        return view('provider.finances.index', [
            'payments' => $payments,
            'expenses' => $expenses,
            'totalPayments' => $totalPayments,
            'totalExpenses' => $totalExpenses,
            'balanceNet' => $balanceNet,
            'startDate' => $startDate,
            'endDate' => $endDate,
            'expenseTypes' => $expenseTypes,
            'allExpenseTypes' => $allExpenseTypes,
            'clients' => $clients,
            'suppliers' => $suppliers,
            'currency' => $currency,
        ]);
    }
}
