<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Http\Requests\PaymentStoreRequest;
use App\Http\Requests\PaymentUpdateRequest;
use App\Models\Payment;
use App\Models\Client;
use App\Models\Appointment;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    public function index(Request $request)
    {
        $provider = auth()->user()->provider;
        
        $query = Payment::where('provider_id', $provider->id)
            ->with(['client', 'appointment'])
            ->orderBy('payment_date', 'desc');

        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }

        if ($request->filled('method')) {
            $query->where('method', $request->method);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('payment_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('payment_date', '<=', $request->date_to);
        }

        $payments = $query->paginate(15);
        $clients = Client::where('provider_id', $provider->id)->orderBy('first_name')->get();

        return view('provider.payments.index', compact('payments', 'clients'));
    }

    public function create(Request $request)
    {
        $provider = auth()->user()->provider;
        $clients = Client::where('provider_id', $provider->id)->orderBy('first_name')->get();
        
        $prefilledClientId = $request->query('client_id');
        $prefilledAppointmentId = $request->query('appointment_id');
        $prefilledAmount = $request->query('amount');

        $appointments = [];
        if ($prefilledClientId) {
            $appointments = Appointment::where('provider_id', $provider->id)
                ->where('client_id', $prefilledClientId)
                ->whereDoesntHave('payments')
                ->orderBy('start_time', 'desc')
                ->get();
        }

        return view('provider.payments.create', compact('clients', 'appointments', 'prefilledClientId', 'prefilledAppointmentId', 'prefilledAmount'));
    }

    public function store(PaymentStoreRequest $request)
    {
        $provider = auth()->user()->provider;
        $currency = $provider->generalSettings['currency'] ?? 'TND';

        $payment = Payment::create([
            'provider_id' => $provider->id,
            'client_id' => $request->client_id,
            'appointment_id' => $request->appointment_ids[0] ?? null,
            'amount' => $request->amount,
            'currency' => $currency,
            'payment_date' => $request->payment_date,
            'method' => $request->method,
            'notes' => $request->notes,
        ]);

        if ($request->appointment_ids) {
            $payment->appointments()->sync($request->appointment_ids);
        }

        return redirect()->route('provider.finances.index', ['tab' => 'factures'])->with('success', 'Encaissement enregistré avec succès.');
    }

    public function edit($payment_uuid)
    {
        $provider = auth()->user()->provider;
        $payment = Payment::where('uuid', $payment_uuid)
            ->where('provider_id', $provider->id)
            ->with('appointments')
            ->firstOrFail();

        $clients = Client::where('provider_id', $provider->id)->orderBy('first_name')->get();
        
        $appointments = [];
        if ($payment->client_id) {
            $currentPaymentAppointmentIds = $payment->appointments->pluck('id')->toArray();
            $appointments = Appointment::where('provider_id', $provider->id)
                ->where('client_id', $payment->client_id)
                ->where(function($query) use ($currentPaymentAppointmentIds) {
                    $query->whereDoesntHave('payments')
                          ->orWhereIn('id', $currentPaymentAppointmentIds);
                })
                ->orderBy('start_time', 'desc')
                ->get();
        }

        return view('provider.payments.edit', compact('payment', 'clients', 'appointments'));
    }

    public function update(PaymentUpdateRequest $request, $payment_uuid)
    {
        $provider = auth()->user()->provider;
        $payment = Payment::where('uuid', $payment_uuid)
            ->where('provider_id', $provider->id)
            ->firstOrFail();

        $payment->update([
            'client_id' => $request->client_id,
            'appointment_id' => $request->appointment_ids[0] ?? null,
            'amount' => $request->amount,
            'payment_date' => $request->payment_date,
            'method' => $request->method,
            'notes' => $request->notes,
        ]);

        if ($request->appointment_ids) {
            $payment->appointments()->sync($request->appointment_ids);
        }

        return redirect()->route('provider.finances.index', ['tab' => 'factures'])->with('success', 'Encaissement modifié avec succès.');
    }

    public function destroy($payment_uuid)
    {
        $provider = auth()->user()->provider;
        $payment = Payment::where('uuid', $payment_uuid)
            ->where('provider_id', $provider->id)
            ->firstOrFail();

        $payment->delete();

        return redirect()->route('provider.finances.index', ['tab' => 'factures'])->with('success', 'Encaissement supprimé avec succès.');
    }

    public function getClientAppointments($client_id)
    {
        $provider = auth()->user()->provider;
        
        $appointments = Appointment::where('provider_id', $provider->id)
            ->where('client_id', $client_id)
            ->whereDoesntHave('payments')
            ->with('service')
            ->orderBy('start_time', 'desc')
            ->get()
            ->map(function($appointment) {
                return [
                    'id' => $appointment->id,
                    'label' => $appointment->start_time->format('d/m/Y H:i') . ' - ' . ($appointment->service->name ?? 'Service') . ' (' . number_format($appointment->price_at_booking, 2) . ' ' . $appointment->currency . ')',
                    'price' => $appointment->price_at_booking
                ];
            });

        return response()->json($appointments);
    }

    public function pdf($payment_uuid)
    {
        $provider = auth()->user()->provider;
        $payment = Payment::where('uuid', $payment_uuid)
            ->where('provider_id', $provider->id)
            ->with(['client', 'appointments.service'])
            ->firstOrFail();

        $pdf = \PDF::loadView('provider.payments.pdf', compact('payment', 'provider'));
        
        return $pdf->download('facture-' . $payment->reference . '.pdf');
    }
}
