<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Http\Requests\ServiceStoreRequest;
use App\Http\Requests\ServiceUpdateRequest;
use App\Models\Service;
use Illuminate\Http\Request;

class ServiceController extends Controller
{
    /**
     * Display a listing of the provider's services.
     */
    public function index(Request $request)
    {
        $provider = auth()->user()->provider;
        $showArchived = $request->boolean('archived');
        
        $query = $showArchived 
            ? $provider->services()->onlyTrashed()
            : $provider->services();
            
        $query->when($request->search, function ($q, $search) {
                $q->where('name', 'like', "%{$search}%");
            })
            ->when($request->has('status'), function ($q) use ($request) {
                if ($request->status === 'active') {
                    $q->where('is_active', true);
                } elseif ($request->status === 'inactive') {
                    $q->where('is_active', false);
                }
            })
            ->orderBy('is_active', 'desc')
            ->orderBy('name', 'asc');

        $services = $query->paginate(15);

        return view('provider.services.index', compact('services', 'showArchived'));
    }

    /**
     * Show the form for creating a new service.
     */
    public function create()
    {
        return view('provider.services.create');
    }

    /**
     * Store a newly created service.
     */
    public function store(ServiceStoreRequest $request)
    {
        $provider = auth()->user()->provider;
        
        $data = $request->validated();
        $data['is_active'] = $request->has('is_active');
        
        // Synchroniser la couleur depuis le color picker ou le champ texte
        if ($request->filled('color_text')) {
            $data['color'] = $request->color_text;
        }
        
        $service = $provider->services()->create($data);

        return redirect()->route('provider.services.index')
            ->with('success', 'Service créé avec succès.');
    }

    /**
     * Show the form for editing the specified service.
     */
    public function edit($service_uuid)
    {
        $service = $this->findServiceByUuid($service_uuid);
        
        return view('provider.services.edit', compact('service'));
    }

    /**
     * Update the specified service.
     */
    public function update(ServiceUpdateRequest $request, $service_uuid)
    {
        $service = $this->findServiceByUuid($service_uuid);
        
        $data = $request->validated();
        $data['is_active'] = $request->has('is_active');
        
        // Synchroniser la couleur depuis le color picker ou le champ texte
        if ($request->filled('color_text')) {
            $data['color'] = $request->color_text;
        }
        
        $service->update($data);

        return redirect()->route('provider.services.index')
            ->with('success', 'Service modifié avec succès.');
    }

    /**
     * Remove the specified service (soft delete).
     */
    public function destroy($service_uuid)
    {
        $service = $this->findServiceByUuid($service_uuid);
        
        $service->delete();

        return redirect()->route('provider.services.index')
            ->with('success', 'Service archivé avec succès.');
    }

    /**
     * Restore an archived service.
     */
    public function restore($service_uuid)
    {
        $provider = auth()->user()->provider;
        
        $service = $provider->services()
            ->onlyTrashed()
            ->where('uuid', $service_uuid)
            ->firstOrFail();
            
        $service->restore();

        return redirect()->route('provider.services.index', ['archived' => 1])
            ->with('success', 'Service restauré avec succès.');
    }

    /**
     * Find service by UUID ensuring it belongs to the authenticated provider.
     */
    private function findServiceByUuid($service_uuid)
    {
        $provider = auth()->user()->provider;
        
        return $provider->services()
            ->where('uuid', $service_uuid)
            ->firstOrFail();
    }
}
