<?php

namespace App\Http\Controllers\Provider;

use App\Http\Controllers\Controller;
use App\Models\Supplier;
use Illuminate\Http\Request;

class SupplierController extends Controller
{
    public function index(Request $request)
    {
        $provider = auth()->user()->provider;
        
        $query = $provider->suppliers()
            ->withCount('expenses')
            ->when($request->search, function ($q, $search) {
                $q->where(function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%{$search}%")
                        ->orWhere('notes', 'like', "%{$search}%");
                });
            })
            ->when($request->phone, function ($q, $phone) {
                $q->where('phone', 'like', "%{$phone}%");
            })
            ->latest('id');
        
        if (!$request->search && !$request->phone) {
            $query->orderBy('name');
        }
        
        $suppliers = $query->get();
        
        return view('provider.suppliers.index', compact('suppliers'));
    }

    public function create()
    {
        return view('provider.suppliers.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'tax_id' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $validated['provider_id'] = auth()->user()->provider->id;
        $validated['is_active'] = $request->has('is_active');
        
        $supplier = Supplier::create($validated);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'supplier' => $supplier
            ]);
        }

        return redirect()->route('provider.suppliers.index')
            ->with('success', 'Fournisseur créé avec succès.');
    }

    public function edit($uuid)
    {
        $supplier = auth()->user()->provider->suppliers()->where('uuid', $uuid)->firstOrFail();
        
        return view('provider.suppliers.edit', compact('supplier'));
    }

    public function update(Request $request, $uuid)
    {
        $supplier = auth()->user()->provider->suppliers()->where('uuid', $uuid)->firstOrFail();
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'tax_id' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $validated['is_active'] = $request->has('is_active');

        $supplier->update($validated);

        return redirect()->route('provider.suppliers.index')
            ->with('success', 'Fournisseur modifié avec succès.');
    }

    public function destroy($uuid)
    {
        $supplier = auth()->user()->provider->suppliers()->where('uuid', $uuid)->firstOrFail();
        $supplier->delete();

        return redirect()->route('provider.suppliers.index')
            ->with('success', 'Fournisseur supprimé avec succès.');
    }
}
