<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class AppointmentStepperRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $provider = auth()->user()->provider;

        return [
            'date' => ['required', 'date'],
            'service_id' => ['required', 'exists:services,id', Rule::exists('services', 'id')->where('provider_id', $provider->id)],
            'client_id' => ['required', 'exists:clients,id', Rule::exists('clients', 'id')->where('provider_id', $provider->id)],
            'start_time' => ['required', 'date_format:H:i'],
            'end_time' => ['required', 'date_format:H:i', 'after:start_time'],
            'price' => ['required', 'numeric', 'min:0'],
            'notes' => ['nullable', 'string'],
            'status' => ['required', 'in:scheduled,complete,cancelled'],
        ];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $provider = auth()->user()->provider;
            $data = $this->validated();

            $startDateTime = $data['date'] . ' ' . $data['start_time'];
            $endDateTime = $data['date'] . ' ' . $data['end_time'];

            // Get appointment ID from route parameter (for update)
            $appointmentUuid = $this->route('appointment_uuid');
            $appointmentId = null;
            
            if ($appointmentUuid) {
                $appointment = $provider->appointments()->where('uuid', $appointmentUuid)->first();
                $appointmentId = $appointment ? $appointment->id : null;
            }

            // Check for overlapping appointments
            $overlapping = $provider->appointments()
                ->when($appointmentId, function ($query, $id) {
                    $query->where('id', '!=', $id);
                })
                ->where(function ($query) use ($startDateTime, $endDateTime) {
                    $query->where(function ($q) use ($startDateTime, $endDateTime) {
                        $q->where('start_time', '<', $endDateTime)
                          ->where('end_time', '>', $startDateTime);
                    });
                })
                ->exists();

            if ($overlapping) {
                $validator->errors()->add('time', 'Ce créneau horaire chevauche un rendez-vous existant.');
            }
        });
    }
}
