<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class AppointmentStoreRequest extends FormRequest
{
    public function authorize(): bool
    {
        return auth()->user()->role === 'provider';
    }

    public function rules(): array
    {
        $provider = auth()->user()->provider;
        
        return [
            'start_time' => [
                'required',
                'date',
                function ($attribute, $value, $fail) use ($provider) {
                    $endTime = $this->input('end_time');
                    if (!$endTime) return;
                    
                    $overlap = $provider->appointments()
                        ->with('client')
                        ->where(function($q) use ($value, $endTime) {
                            $q->where(function($q1) use ($value, $endTime) {
                                $q1->where('start_time', '<', $endTime)
                                   ->where('end_time', '>', $value);
                            });
                        })
                        ->first();
                    
                    if ($overlap) {
                        $clientName = $overlap->client ? $overlap->client->full_name : 'Client non spécifié';
                        $start = \Carbon\Carbon::parse($overlap->start_time)->format('H:i');
                        $end = \Carbon\Carbon::parse($overlap->end_time)->format('H:i');
                        $fail("Ce créneau chevauche un rendez-vous existant : {$clientName} de {$start} à {$end}");
                    }
                }
            ],
            'end_time' => 'required|date|after:start_time',
            'client_id' => [
                'nullable',
                'exists:clients,id',
                function ($attribute, $value, $fail) use ($provider) {
                    if ($value && !$provider->clients()->where('id', $value)->exists()) {
                        $fail('Le client sélectionné n\'appartient pas à votre compte.');
                    }
                }
            ],
            'service_id' => [
                'nullable',
                'exists:services,id',
                function ($attribute, $value, $fail) use ($provider) {
                    if ($value && !$provider->services()->where('id', $value)->exists()) {
                        $fail('Le service sélectionné n\'appartient pas à votre compte.');
                    }
                }
            ],
            'price_at_booking' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'status' => 'required|in:scheduled,completed,canceled,no_show',
            'notes' => 'nullable|string',
        ];
    }

    public function messages(): array
    {
        return [
            'start_time.required' => 'L\'heure de début est obligatoire.',
            'start_time.date' => 'L\'heure de début doit être une date valide.',
            'end_time.required' => 'L\'heure de fin est obligatoire.',
            'end_time.date' => 'L\'heure de fin doit être une date valide.',
            'end_time.after' => 'L\'heure de fin doit être après l\'heure de début.',
            'client_id.exists' => 'Le client sélectionné n\'existe pas.',
            'service_id.exists' => 'Le service sélectionné n\'existe pas.',
            'price_at_booking.numeric' => 'Le prix doit être un nombre.',
            'price_at_booking.min' => 'Le prix doit être positif.',
            'status.required' => 'Le statut est obligatoire.',
            'status.in' => 'Le statut sélectionné n\'est pas valide.',
        ];
    }
}