<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Appointment extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'uuid',
        'provider_id',
        'client_id',
        'service_id',
        'start_time',
        'end_time',
        'price_at_booking',
        'currency',
        'status',
        'notes',
        'sync_status',
    ];

    protected $casts = [
        'start_time' => 'datetime',
        'end_time' => 'datetime',
        'price_at_booking' => 'decimal:2',
        'status' => 'string',
    ];

    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETE = 'complete';
    const STATUS_LATE = 'late';
    const STATUS_CANCELLED = 'cancelled';

    public function getStatusColorClass(): string
    {
        return match($this->status) {
            self::STATUS_LATE => 'text-red-600 bg-red-50',
            self::STATUS_COMPLETE => 'text-green-600 bg-green-50',
            self::STATUS_IN_PROGRESS => 'text-orange-600 bg-orange-50',
            self::STATUS_SCHEDULED => 'text-blue-600 bg-blue-50',
            default => 'text-gray-600 bg-gray-50'
        };
    }

    public function getStatusLabel(): string
    {
        return match($this->status) {
            self::STATUS_LATE => 'EN RETARD',
            self::STATUS_COMPLETE => 'TERMINÉ',
            self::STATUS_IN_PROGRESS => 'EN COURS',
            self::STATUS_SCHEDULED => 'PLANIFIÉ',
            default => 'INCONNU'
        };
    }

    public function getDurationInHours(): float
    {
        if (!$this->start_time || !$this->end_time) {
            return 0;
        }
        return $this->start_time->diffInMinutes($this->end_time) / 60;
    }

    public function getClientInitials(): string
    {
        if (!$this->client) {
            return 'XX';
        }
        $names = explode(' ', trim($this->client->first_name . ' ' . $this->client->last_name));
        return strtoupper(substr($names[0], 0, 1) . (isset($names[1]) ? substr($names[1], 0, 1) : ''));
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }
            if (empty($model->sync_status)) {
                $model->sync_status = 'synced';
            }
        });
    }

    public function provider(): BelongsTo
    {
        return $this->belongsTo(Provider::class);
    }

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function service(): BelongsTo
    {
        return $this->belongsTo(Service::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }
}