<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Client extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'uuid',
        'provider_id',
        'first_name',
        'last_name',
        'phone',
        'email',
        'address',
        'gender',
        'birth_date',
        'notes',
        'sync_status',
        'last_visit_at',
    ];

    protected $casts = [
        'birth_date' => 'date',
        'last_visit_at' => 'datetime',
    ];

    /**
     * Boot function to auto-generate UUID on creation.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }
            if (empty($model->sync_status)) {
                $model->sync_status = 'synced';
            }
        });
    }

    /**
     * Get the provider that owns the client.
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(Provider::class);
    }

    /**
     * Get the full name of the client.
     */
    public function getFullNameAttribute(): string
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    // TODO: Relations pour les futurs modules
    // Ces relations seront utilisées lorsque les modules Séances et Paiements seront développés
    
    /**
     * Get the appointments for the client.
     * TODO: Activer quand le module Appointments sera créé
     */
    // public function appointments(): HasMany
    // {
    //     return $this->hasMany(Appointment::class);
    // }

    /**
     * Get the payments for the client.
     * TODO: Activer quand le module Payments sera créé
     */
    // public function payments(): HasMany
    // {
    //     return $this->hasMany(Payment::class);
    // }

    /**
     * Get the expenses for the client.
     */
    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class);
    }

    /**
     * Get the appointments for the client.
     */
    public function appointments(): HasMany
    {
        return $this->hasMany(Appointment::class);
    }

    /**
     * Get the payments for the client.
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }
}
