<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Payment extends Model
{
    protected $fillable = [
        'uuid',
        'provider_id',
        'client_id',
        'appointment_id',
        'amount',
        'currency',
        'payment_date',
        'method',
        'reference',
        'notes',
        'sync_status',
    ];

    protected $casts = [
        'payment_date' => 'date',
        'amount' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }
            if (empty($model->sync_status)) {
                $model->sync_status = 'synced';
            }
            if (empty($model->reference)) {
                $prefix = $model->provider->billingSettings['invoice_prefix'] ?? 'PAY';
                $model->reference = $prefix . '-' . date('Ymd') . '-' . strtoupper(Str::random(6));
            }
            if (empty($model->currency) && $model->provider) {
                $model->currency = $model->provider->generalSettings['currency'] ?? 'TND';
            }
        });
    }

    public function provider(): BelongsTo
    {
        return $this->belongsTo(Provider::class);
    }

    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    public function appointment(): BelongsTo
    {
        return $this->belongsTo(Appointment::class);
    }

    public function appointments()
    {
        return $this->belongsToMany(Appointment::class, 'appointment_payment');
    }
}
