<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Provider extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array<string>
     */
    protected $fillable = [
        'uuid',
        'user_id',
        'business_name',
        'phone',
        'address',
        'city',
        'country',
        'logo',
        'description',
        'tax_id',
        'legal_name',
        'working_hours',
        'settings',
        'sync_status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'working_hours' => 'array',
        'settings' => 'array',
    ];

    /**
     * Get general settings.
     */
    public function getGeneralSettingsAttribute()
    {
        return $this->settings['general'] ?? [];
    }

    /**
     * Get calendar settings.
     */
    public function getCalendarSettingsAttribute()
    {
        return $this->settings['calendar'] ?? [];
    }

    /**
     * Get billing settings.
     */
    public function getBillingSettingsAttribute()
    {
        return $this->settings['billing'] ?? [];
    }

    /**
     * Update general settings.
     */
    public function updateGeneralSettings(array $generalSettings)
    {
        $settings = $this->settings ?? [];
        $settings['general'] = array_merge($settings['general'] ?? [], $generalSettings);
        $this->update(['settings' => $settings]);
    }

    /**
     * Update calendar settings.
     */
    public function updateCalendarSettings(array $calendarSettings)
    {
        $settings = $this->settings ?? [];
        $settings['calendar'] = array_merge($settings['calendar'] ?? [], $calendarSettings);
        $this->update(['settings' => $settings]);
    }

    /**
     * Update billing settings.
     */
    public function updateBillingSettings(array $billingSettings)
    {
        $settings = $this->settings ?? [];
        $settings['billing'] = array_merge($settings['billing'] ?? [], $billingSettings);
        $this->update(['settings' => $settings]);
    }

    /**
     * Boot function to auto-generate UUID on creation.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }
            if (empty($model->sync_status)) {
                $model->sync_status = 'synced';
            }
        });
    }

    /**
     * Get the user that owns the provider.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the clients for the provider.
     */
    public function clients(): HasMany
    {
        return $this->hasMany(Client::class);
    }

    /**
     * Get the services for the provider.
     */
    public function services(): HasMany
    {
        return $this->hasMany(Service::class);
    }

    /**
     * Get the expense types for the provider.
     */
    public function expenseTypes(): HasMany
    {
        return $this->hasMany(ExpenseType::class);
    }

    /**
     * Get the expenses for the provider.
     */
    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class);
    }

    /**
     * Get the appointments for the provider.
     */
    public function appointments(): HasMany
    {
        return $this->hasMany(Appointment::class);
    }

    /**
     * Get the payments for the provider.
     */
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the suppliers for the provider.
     */
    public function suppliers(): HasMany
    {
        return $this->hasMany(Supplier::class);
    }
}
