@extends('layouts.provider')

@section('title', 'Nouveau Rendez-vous')
@section('page-title', 'Nouveau Rendez-vous')

@section('content')
<div class="space-y-6">
    <x-card>
        <x-slot name="header">
            <div class="flex items-center gap-3">
                <a href="{{ route('provider.appointments.calendar') }}" class="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-300">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                    </svg>
                </a>
                <h2 class="text-xl font-bold text-gray-900 dark:text-white">Nouveau Rendez-vous</h2>
            </div>
        </x-slot>

        <!-- Formulaire -->
        <form action="{{ route('provider.appointments.store') }}" method="POST" class="space-y-6" onsubmit="return prepareSubmit()">
            @csrf
            
            <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">Informations du rendez-vous</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <!-- Date de début -->
                <div>
                    <label for="start_date" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Date <span class="text-red-500">*</span>
                    </label>
                    <input type="date" 
                           id="start_date" 
                           name="start_date" 
                           value="{{ old('start_date', $defaultDate) }}"
                           required
                           class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                </div>

                <!-- Heure de début -->
                <div>
                    <label for="start_time_input" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Heure de début <span class="text-red-500">*</span>
                    </label>
                    <input type="text" 
                           id="start_time_input" 
                           value="{{ old('start_time_input', $defaultTime) }}"
                           placeholder="09:00"
                           pattern="[0-2][0-9]:[0-5][0-9]"
                           required
                           class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('start_time') border-red-500 @enderror">
                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Format: HH:MM (ex: 09:00, 15:30)</p>
                    @error('start_time')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Heure de fin -->
                <div>
                    <label for="end_time_input" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Heure de fin <span class="text-red-500">*</span>
                    </label>
                    <input type="text" 
                           id="end_time_input" 
                           value="{{ old('end_time_input') }}"
                           placeholder="10:00"
                           pattern="[0-2][0-9]:[0-5][0-9]"
                           required
                           class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('end_time') border-red-500 @enderror">
                    <p class="mt-1 text-xs text-gray-500">Format: HH:MM (ex: 10:00, 16:30)</p>
                    @error('end_time')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
                
                <input type="hidden" id="start_time" name="start_time">
                <input type="hidden" id="end_time" name="end_time">

                <!-- Client -->
                <div>
                    <label for="client_id" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Client
                    </label>
                    <select id="client_id" 
                            name="client_id" 
                            class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('client_id') border-red-500 @enderror">
                        <option value="">Sélectionner un client</option>
                        @foreach($clients as $client)
                            <option value="{{ $client->id }}" {{ old('client_id') == $client->id ? 'selected' : '' }} class="bg-blue-50">
                                {{ $client->full_name }}
                            </option>
                        @endforeach
                    </select>
                    @error('client_id')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Service -->
                <div>
                    <label for="service_id" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Service
                    </label>
                    <select id="service_id" 
                            name="service_id" 
                            class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('service_id') border-red-500 @enderror">
                        <option value="">Sélectionner un service</option>
                        @foreach($services as $service)
                            <option value="{{ $service->id }}" 
                                    data-price="{{ $service->base_price }}"
                                    data-currency="{{ $service->currency }}"
                                    data-duration="{{ $service->duration_minutes }}"
                                    {{ old('service_id') == $service->id ? 'selected' : '' }}>
                                {{ $service->name }} - {{ $service->formatted_price }} ({{ $service->formatted_duration }})
                            </option>
                        @endforeach
                    </select>
                    @error('service_id')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Prix -->
                <div>
                    <label for="price_at_booking" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Prix
                    </label>
                    <div class="relative">
                        <input type="number" 
                               id="price_at_booking" 
                               name="price_at_booking" 
                               value="{{ old('price_at_booking') }}"
                               step="0.01"
                               min="0"
                               class="w-full px-4 py-3 pr-16 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('price_at_booking') border-red-500 @enderror"
                               placeholder="0.00">
                        <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                            <span class="text-gray-500 dark:text-gray-400 text-sm">TND</span>
                        </div>
                    </div>
                    @error('price_at_booking')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Alerte ajustement de prix -->
                <div class="md:col-span-2">
                    <div id="priceAlert" class="hidden bg-amber-50 border-l-4 border-amber-500 p-4 rounded-r-lg">
                        <div class="flex items-start justify-between">
                            <div class="flex items-start flex-1">
                                <div class="flex-shrink-0">
                                    <svg class="h-5 w-5 text-amber-500" fill="currentColor" viewBox="0 0 20 20">
                                        <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                                    </svg>
                                </div>
                                <div class="ml-3">
                                    <p class="text-sm font-medium text-amber-800" id="priceAlertMessage"></p>
                                    <p class="mt-1 text-sm text-amber-700" id="priceAlertQuestion"></p>
                                    <div class="mt-2 text-xs font-mono bg-amber-100 text-amber-900 px-3 py-2 rounded inline-block" id="priceAlertCalculation"></div>
                                </div>
                            </div>
                            <div class="flex gap-2 ml-4">
                                <button type="button" onclick="acceptPriceAdjustment()" class="bg-amber-600 hover:bg-amber-700 text-white text-sm font-medium py-2 px-4 rounded transition-colors whitespace-nowrap">
                                    ✓ Adapter
                                </button>
                                <button type="button" onclick="rejectPriceAdjustment()" class="bg-gray-200 hover:bg-gray-300 text-gray-800 text-sm font-medium py-2 px-4 rounded transition-colors whitespace-nowrap">
                                    ✗ Garder
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Statut -->
                <div>
                    <label for="status" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Statut <span class="text-red-500">*</span>
                    </label>
                    <select id="status" 
                            name="status" 
                            required
                            class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('status') border-red-500 @enderror">
                        <option value="scheduled" {{ old('status', 'scheduled') === 'scheduled' ? 'selected' : '' }}>Programmé</option>
                        <option value="completed" {{ old('status') === 'completed' ? 'selected' : '' }}>Terminé</option>
                        <option value="canceled" {{ old('status') === 'canceled' ? 'selected' : '' }}>Annulé</option>
                        <option value="no_show" {{ old('status') === 'no_show' ? 'selected' : '' }}>Absent</option>
                    </select>
                    @error('status')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Notes -->
                <div class="md:col-span-2">
                    <label for="notes" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        Notes
                    </label>
                    <textarea id="notes" 
                              name="notes" 
                              rows="3"
                              class="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 @error('notes') border-red-500 @enderror"
                              placeholder="Notes optionnelles sur ce rendez-vous...">{{ old('notes') }}</textarea>
                    @error('notes')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>
            </div>

            <!-- Actions -->
            <div class="flex gap-3 pt-6 border-t border-gray-200 dark:border-gray-700">
                <x-button type="submit" class="flex-1">Créer le rendez-vous</x-button>
                <x-button href="{{ route('provider.appointments.calendar') }}" variant="secondary">Annuler</x-button>
            </div>
        </form>
    </x-card>

<script>
    let serviceDurationMinutes = 0;
    let serviceBasePrice = 0;
    let pendingNewPrice = null;
    let priceAlertAnswered = false;
    let allowSubmit = false;
    let endTimeSetManually = false;

    function calculateEndTime() {
        if (endTimeSetManually) return;
        
        const startTime = document.getElementById('start_time_input').value;
        const serviceSelect = document.getElementById('service_id');
        const selectedOption = serviceSelect.options[serviceSelect.selectedIndex];
        const duration = selectedOption.getAttribute('data-duration');
        
        if (startTime && duration && startTime.match(/^\d{2}:\d{2}$/)) {
            const [hours, minutes] = startTime.split(':').map(Number);
            const startMinutes = hours * 60 + minutes;
            const endMinutes = startMinutes + parseInt(duration);
            
            const endHours = Math.floor(endMinutes / 60);
            const endMins = endMinutes % 60;
            
            const endTime = String(endHours).padStart(2, '0') + ':' + String(endMins).padStart(2, '0');
            document.getElementById('end_time_input').value = endTime;
        }
    }

    document.getElementById('service_id').addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const price = selectedOption.getAttribute('data-price');
        const duration = selectedOption.getAttribute('data-duration');
        
        serviceDurationMinutes = duration ? parseInt(duration) : 0;
        serviceBasePrice = price ? parseFloat(price) : 0;
        
        if (price) {
            document.getElementById('price_at_booking').value = price;
        }
        
        if (duration && !endTimeSetManually) {
            calculateEndTime();
        }
        
        hidePriceAlert();
    });
    
    document.getElementById('start_date').addEventListener('change', checkForPriceAdjustment);
    document.getElementById('start_time_input').addEventListener('change', function() {
        setTimeout(checkForPriceAdjustment, 100);
    });
    
    document.getElementById('end_time_input').addEventListener('change', function() {
        endTimeSetManually = true;
        checkForPriceAdjustment();
    });
    


    function checkForPriceAdjustment() {
        if (!serviceDurationMinutes || !serviceBasePrice) return;
        
        const startTime = document.getElementById('start_time_input').value;
        const endTime = document.getElementById('end_time_input').value;
        
        if (!startTime || !endTime || !startTime.match(/^\d{2}:\d{2}$/) || !endTime.match(/^\d{2}:\d{2}$/)) return;
        
        const [startH, startM] = startTime.split(':').map(Number);
        const [endH, endM] = endTime.split(':').map(Number);
        const sessionDurationMinutes = (endH * 60 + endM) - (startH * 60 + startM);
        
        if (sessionDurationMinutes <= 0) {
            hidePriceAlert();
            return;
        }
        
        if (sessionDurationMinutes === serviceDurationMinutes) {
            hidePriceAlert();
            return;
        }
        
        const ratio = sessionDurationMinutes / serviceDurationMinutes;
        const newPrice = (serviceBasePrice * ratio).toFixed(2);
        
        showPriceAlert(newPrice, ratio, sessionDurationMinutes);
    }

    function showPriceAlert(newPrice, ratio, sessionDurationMinutes) {
        pendingNewPrice = newPrice;
        priceAlertAnswered = false;
        
        const alertDiv = document.getElementById('priceAlert');
        const messageEl = document.getElementById('priceAlertMessage');
        const questionEl = document.getElementById('priceAlertQuestion');
        const calculationEl = document.getElementById('priceAlertCalculation');
        
        const adjustmentType = sessionDurationMinutes > serviceDurationMinutes ? 'augmentée' : 'réduite';
        
        messageEl.textContent = `Durée ${adjustmentType} : ${sessionDurationMinutes} min (au lieu de ${serviceDurationMinutes} min)`;
        questionEl.textContent = `Voulez-vous adapter le prix à ${newPrice} TND ?`;
        calculationEl.textContent = `${serviceBasePrice} TND × ${ratio.toFixed(2)} = ${newPrice} TND`;
        
        alertDiv.classList.remove('hidden');
        alertDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }

    function hidePriceAlert() {
        document.getElementById('priceAlert').classList.add('hidden');
        pendingNewPrice = null;
    }

    function acceptPriceAdjustment() {
        if (pendingNewPrice) {
            document.getElementById('price_at_booking').value = pendingNewPrice;
        }
        priceAlertAnswered = true;
        hidePriceAlert();
    }

    function rejectPriceAdjustment() {
        priceAlertAnswered = true;
        hidePriceAlert();
    }

    function prepareSubmit() {
        const startDate = document.getElementById('start_date').value;
        const startTimeInput = document.getElementById('start_time_input').value;
        const endTimeInput = document.getElementById('end_time_input').value;
        
        document.getElementById('start_time').value = startDate + ' ' + startTimeInput + ':00';
        document.getElementById('end_time').value = startDate + ' ' + endTimeInput + ':00';
        
        const alertVisible = !document.getElementById('priceAlert').classList.contains('hidden');
        
        if (alertVisible && !priceAlertAnswered) {
            allowSubmit = true;
            return false;
        }
        
        return true;
    }
</script>
        </div>
    </div>
@endsection
